<?php
namespace Modules\Gig\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Modules\Core\Events\CreatedServicesEvent;
use Modules\Core\Events\UpdatedServiceEvent;
use Modules\FrontendController;
use Modules\Gig\Models\Gig;
use Modules\Core\Models\Attributes;
use Modules\Gig\Models\GigCategory;
use Modules\Gig\Models\GigTerm;
use Modules\Gig\Models\GigTranslation;

class GigManageController extends FrontendController
{

    public function callAction($method, $parameters)
    {
        if(!Gig::isEnable())
        {
            return redirect('/');
        }
        return parent::callAction($method, $parameters); // TODO: Change the autogenerated stub
    }

    public function index(Request $request){

        $this->checkPermission('gig_manage');

        $query = Gig::query() ;
        $query->orderBy('id', 'desc');
        if (!empty($s = $request->input('s'))) {
            $query->where('title', 'LIKE', '%' . $s . '%');
            $query->orderBy('title', 'asc');
        }

        if ($this->hasPermission('gig_manage_others')) {
            if (!empty($author = $request->input('vendor_id'))) {
                $query->where('author_id', $author);
            }
        } else {
            $query->where('author_id', Auth::id());
        }

        $rows = $query->with(['author','cat','cat2','cat3'])->paginate(20);

        $data = [
            'rows' => $rows,
            'page_title' => __("All Gigs"),
            'menu_active' => 'seller_gigs'
        ];
        return view('Gig::frontend.seller.manageGig.index', $data);
    }

    public function edit(Request $request, $id){
        $this->checkPermission('gig_manage');
        $row = Gig::query()->where('author_id', Auth::id())->find($id);
        if (empty($row)){
            return redirect(route('seller.all.gigs'));
        }

        $translation = $row->translateOrOrigin($request->query('lang'));

        $data = [
            'row' => $row,
            'translation' => $translation,
            "selected_terms" => $row->terms->pluck('term_id'),
            'attributes'     => Attributes::where('service', 'gig')->get(),
            'enable_multi_lang'=>true,
            'tags' => $row->getTags(),
            'page_title' => __("Edit gig"),
            'menu_active' => 'seller_gigs',
            'is_user_page' => true
        ];

        return view('Gig::frontend.seller.manageGig.edit', $data);
    }

    public function create(Request $request){
        $this->checkPermission('gig_manage');
        if(!is_admin() && !auth()->user()->checkGigPostingMaximum()){
            return redirect(route('gig.admin.index'))->with('error', __('You posted the maximum number of gigs') );
        }
        $row = new Gig();
        $row->fill([
            'status' => 'publish'
        ]);
        $data = [
            'row'            => $row,
            'attributes'     => Attributes::where('service', 'gig')->get(),
            'categories' => GigCategory::query()->get()->toTree(),
            'translation'    => new GigTranslation(),
            'page_title'     => __("Add new Gig"),
            'is_user_page' => true
        ];
        return view('Gig::frontend.seller.manageGig.edit', $data);
    }

    public function store(Request $request, $id){
        if($id>0){
            $this->checkPermission('gig_manage');
            $row = Gig::query()->where('author_id', Auth::id())->find($id);
            if (empty($row)) {
                return redirect(route('seller.all.gigs'));
            }
        }else{
            $this->checkPermission('gig_manage');
            $row = new Gig();
            $row->status = "publish";
        }

        $rules = [
            'title'=>'required',
            'cat_id'=>'required',
            'packages.0.name'=>'required',
            'basic_price'=>'required',
            'packages.0.desc'=>'required',
            'packages.0.delivery_time'=>'required',
            'packages.0.revision'=>'required',
        ];
        if(is_default_lang()){
            $request->validate($rules,[
                'title.required'=>__('Gig title is required'),
                'cat_id.required'=>__("Category is required"),
                'packages.0.name.required'=>__("Basic package name is required"),
                'basic_price.required'=>__("Basic price is required"),
                'packages.0.desc.required'=>__("Basic package description is required"),
                'packages.0.delivery_time.required'=>__("Basic delivery time is required"),
                'packages.0.revision.required'=>__("Basic revision is required"),
            ]);
        }

        $dataKeys = [
            'title',
            'content',
            'status',
            'is_featured',
            'cat_id',
            'cat2_id',
            'cat3_id',

            'video_url',
            'faqs',
            'image_id',
            'gallery',

            'extra_price',

            'basic_price',
            'standard_price',
            'premium_price',

            'packages',
            'package_compare',
            'requirements',
        ];

        $row->fillByAttr($dataKeys,$request->input());
        $row->author_id = Auth::id();

        if($request->input('slug')){
            $row->slug = $request->input('slug');
        }

        if($packages = $request->input('packages') and !empty($packages[0]['delivery_time']) ){
            $row->basic_delivery_time = $packages[0]['delivery_time'];
        }else{
            $row->basic_delivery_time = 0;
        }

        $res = $row->saveOriginOrTranslation($request->input('lang'),true);
        if ($res) {
            if(!$request->input('lang') or is_default_lang($request->input('lang'))) {
                $this->saveTerms($row, $request);
            }
            if(is_default_lang($request->query('lang'))){
                $row->saveTag($request->input('tag_name'), $request->input('tag_ids'));
            }

            if($id > 0 ){
                event(new UpdatedServiceEvent($row));

                return back()->with('success',  __('Gig updated') );
            }else{
                event(new CreatedServicesEvent($row));

                return redirect(route('seller.gig.edit',$row->id))->with('success', __('Gig created') );
            }
        }
    }

    public function delete(Request $request){
        $this->checkPermission('gig_manage');

        $row = Gig::query()->where('author_id', Auth::id())->find($request->get('gig_id'));
        if (empty($row)){
            return redirect(route('seller.all.gigs'));
        }

        $row->delete();
        event(new UpdatedServiceEvent($row));

        return redirect()->back()->with('success', __('Deleted success!'));

    }

    public function saveTerms($row, $request)
    {
        $this->checkPermission('gig_manage');
        if (empty($request->input('terms'))) {
            GigTerm::where('target_id', $row->id)->delete();
        } else {
            $term_ids = $request->input('terms');
            foreach ($term_ids as $term_id) {
                GigTerm::firstOrCreate([
                    'term_id' => $term_id,
                    'target_id' => $row->id
                ]);
            }
            GigTerm::where('target_id', $row->id)->whereNotIn('term_id', $term_ids)->delete();
        }
    }
}
